module clipboardview

//	**************************************************************************************************
//
//	A program that can show and set the current content of the clipboard.
//
//	The program has been written in Clean 1.3.1 and uses the Clean Standard Object I/O library 1.1
//	
//	**************************************************************************************************

import StdEnv													// Import all standard library modules
import StdIO													// Import all standard gui library modules

::	NoState														// NoState is a simple singleton type constructor
	=	NoState

Start :: *World -> *World										// The main rule
Start world
	#	(ids,world)	= openIds 3 world							// Create 3 Id values
	#	world		= startIO									// Create and evaluate an interactive process which has:
						NoState									//    no local  process state
						NoState									//    no public process state
						[initialise ids]						//    one initialisation action
						[]										//    only default attributes
						world
	=	world

initialise ids process
	#	(error,process)	= openDialog NoState clipview process	// Open the clipview dialog
	|	error<>NoError											// In case of an error:
		=	closeProcess process								//    terminate the interactive process
	|	otherwise												// Otherwise:
		=	process												//    initialisation is complete
where
	viewid	= ids!!0
	showid	= ids!!1
	setid	= ids!!2
	
	clipview
		= Dialog												// The clipview window is a Dialog
			"Clipboard Viewer"									// The title of the clipview window
			(													// The controls of the clipview window:
				showclip										//    the content display and refresh button
			:+:													//    +
				setclip											//    the edit display and set button
			:+:													//    +
				quit											//    the quit button
			)
			[	WindowId viewid									// The identification of the clipview window
			]
	
	showclip
		=   EditControl "" width nrlines
				[	ControlSelectState Unable
				,	ControlId          showid
				,	ControlPos         (Left,zero)
				]
		:+: ButtonControl "Show"
				[	ControlFunction    (noLS show)
				]
	
	setclip
		=   EditControl "" width nrlines
				[	ControlId          setid
				,	ControlPos         (Left,zero)
				]
		:+: ButtonControl "Set"
				[	ControlFunction    (noLS set)
				]
	
	quit=   ButtonControl "Quit"
				[	ControlFunction    (noLS closeProcess)
				,	ControlPos         (Center,zero)
				]
	
	width	= hmm 100.0
	nrlines	= 5
	
	show process
		#	(content,process)	= getClipboard process
			text				= getString content
		=	appPIO (setWindow viewid [setControlTexts [(showid,text)]]) process
	
	set process
		#	(dialog,process)	= accPIO (getWindow viewid) process
			text				= fromJust (snd (hd (getControlTexts [setid] (fromJust dialog))))
		=	setClipboard [toClipboard text] process
	
	getString [clip:clips]
		|	isNothing item
			=	getString clips
		|	otherwise
			=	fromJust item
	where
		item	= fromClipboard clip
	getString []
		=	""
